<?php
/**
 * Post Functions
 *
 * @package     AutomatorWP\Restrict_Content\Post_Functions
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Helper function to easily get a AutomatorWP Restrict Content meta
 *
 * @since 1.0.0
 *
 * @param integer   $post_id
 * @param string    $meta_key
 * @param bool      $single
 * @return mixed
 */
function automatorwp_restrict_content_get_post_meta( $post_id, $meta_key, $single = true ) {

    if( $post_id === null ) {
        $post_id = get_the_ID();
    }

    $prefix = '_automatorwp_restrict_content_';

    return get_post_meta( $post_id, $prefix . $meta_key, $single );

}

/**
 * Return true if post is restricted and has restrictions
 *
 * @since 1.0.0
 *
 * @param integer $post_id
 *
 * @return bool
 */
function automatorwp_restrict_content_is_post_restricted( $post_id = null ) {

    $restricted = automatorwp_restrict_content_get_post_meta( $post_id, 'restrict' );

    // Post restricted if enabled
    return $restricted === 'on';
}

/**
 * Return true if post is restricted and has access restricted
 *
 * @since 1.0.0
 *
 * @param integer $post_id
 *
 * @return bool
 */
function automatorwp_restrict_content_is_post_access_restricted( $post_id = null ) {

    if( ! automatorwp_restrict_content_is_post_restricted( $post_id ) ) {
        return false;
    }

    $restriction = automatorwp_restrict_content_get_post_meta( $post_id, 'restriction' );

    return ( in_array( $restriction, array( 'redirect_page', 'redirect_url' ) ) );

}

/**
 * Return true if post is restricted and has content restricted
 *
 * @since 1.0.0
 *
 * @param integer $post_id
 *
 * @return bool
 */
function automatorwp_restrict_content_is_post_content_restricted( $post_id = null ) {

    if( ! automatorwp_restrict_content_is_post_restricted( $post_id ) ) {
        return false;
    }

    $restriction = automatorwp_restrict_content_get_post_meta( $post_id, 'restriction' );

    return ( in_array( $restriction, array( 'content_replacement' ) ) );

}

/**
 * Check if user has been granted to access to a given post
 *
 * Important! Administrator and authors are not restricted to access
 *
 * @since 1.0.0
 *
 * @param integer $post_id
 * @param integer $user_id
 *
 * @return bool
 */
function automatorwp_restrict_content_is_user_granted_to_post( $post_id = null, $user_id = null ) {

    if( $post_id === null ) {
        $post_id = get_the_ID();
    }

    if( $user_id === null ) {
        $user_id = get_current_user_id();
    }

    // Required Roles
    $required_roles = automatorwp_restrict_content_get_post_meta( $post_id, 'required_roles', false );

    if( is_array( $required_roles ) && count( $required_roles ) ) {

        // Change init value since this option has been configured
        $meet_role = false;

        foreach( $required_roles as $required_role ) {
            if( user_can( $user_id, $required_role[0] ) ) {
                $meet_role = true;
            }
        }

        // Access denied
        if( ! $meet_role ) {
            return false;
        }

    }

    // Required Access Levels
    $required_access_levels = automatorwp_restrict_content_get_post_meta( $post_id, 'required_access_levels', false );

    if( is_array( $required_access_levels ) && count( $required_access_levels ) ) {

        // Change init value since this option has been configured
        $meet_access_level = false;

        $user_access_levels = automatorwp_restrict_content_get_user_access_levels( $user_id );

        foreach( $user_access_levels as $user_access_level ) {

            foreach( $required_access_levels as $required_access_level ) {

                if( absint( $required_access_level[0] ) === absint( $user_access_level->id ) ) {
                    $meet_access_level = true;
                }
            }

        }

        // Access denied
        if( ! $meet_access_level ) {
            return false;
        }

    }

    /**
     * Filter to custom check is user has access to a post
     *
     * @since 1.0.0
     *
     * @param bool  $access_granted Whatever is granted to access to this post or not
     * @param int   $post_id        The post ID
     * @param int   $user_id        The user ID
     */
    return apply_filters( 'automatorwp_restrict_content_is_user_granted_to_post', true, $post_id, $user_id );

}